/****************************************************************************
 *   $Id:: main.c 2010-11-09 02:23:44Z nxp12832		                 		$
 *   Project: NXP Power Profiles Usage Example
 *
 *   Description:
 *     This file contains PLL setting demo using the boot-ROM API calls.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/

#include "LPC11xx.h"
#include "power_api.h"
#include "power_profiles.h"
#include "uart.h"
#include <stdio.h>

#define UART_BAUD		57600
#define ENABLE_CLKOUT 	0
#define SYS_OSC  		1

/*	Variable set by SystemInit() and used by the UART driver */
extern uint32_t SystemAHBFrequency;
extern volatile uint32_t UARTCount;
extern volatile uint8_t  UARTBuffer[];

/*	Function used by Retarget.c for printf statements */
int  schar(int ch);
int32_t  gchar(void);

POWER_PROFILE profile;

struct entry {
	uint8_t		char_selection;
	uint32_t	freq_selection;
};

const struct entry freq_lookup[] = {
	/* Character Selection, Frequency to be used */
	{'0', 	 1000000UL},
	{'1', 	 2000000UL},
	{'2', 	 4000000UL},
	{'3', 	 6000000UL},
	{'4',   10000000UL},
	{'5', 	12000000UL},
	{'6', 	15000000UL},
	{'7', 	20000000UL},
	{'8', 	24000000UL},
	{'9', 	30000000UL},
	{'a', 	36000000UL},
	{'b', 	40000000UL},
	{'c', 	48000000UL},
	{'d', 	50000000UL},

	{'\0',	12000000UL}		//Default selection if no match is found
};

void print_menu(void){
	uint32_t i;

	printf("\x1b#8");	//VT100 Screen align
	printf("\x1b[2J");	//VT100 Clear screen
#if SYS_OSC
	printf("set_pll Power Profiles Demo LPC1100 - MAIN OSC\n\r");
#else
	printf("set_pll Power Profiles Demo LPC1100 - IRC OSC\n\r");
#endif /* SYS_OSC */
	printf("Press any digit from '0' - 'd' to change the system frequency\n\r");
	for(i = 0; freq_lookup[i].char_selection != '\0'; i++){
		printf("    '%c': %8d Hz\n\r", freq_lookup[i].char_selection, freq_lookup[i].freq_selection);
	}
}

uint32_t lookup_freq(uint8_t selection) {
	uint32_t i;
	
	/*	Search the lookup table until character selection is found
		or return the default frequency value for the default selection */
	for(i = 0; freq_lookup[i].char_selection != '\0'; i++){
		/*	If match found, return the freq for that entry */
		if(freq_lookup[i].char_selection == selection)
			return freq_lookup[i].freq_selection;
	}
	/*	No match found, return the default */
	return freq_lookup[i].freq_selection;
}



int main(void){
	uint32_t counter = 0;
	volatile static int32_t cmd;

	/*	Enable AHB/APB bridge; ROM; RAM; FLASHREG; FLASHARRAY; GPIO; IOCON */
	LPC_SYSCON->SYSAHBCLKCTRL = (1<<0)|(1<<1)|(1<<2)|(1<<3)|(1<<4)|(1<<6)|(1<<16);

#if SYS_OSC	
	/*	Enable system oscillator and go into a little dummy delay */
	LPC_SYSCON->SYSOSCCTRL = 0x00;						//No Bypass; 1-20Mhz Range
	LPC_SYSCON->PDRUNCFG &= ~(0x1<<5);					//Enable system oscillator
	for ( counter = 0; counter < 0x100; counter++ );	//Dummy wait for the system osc

	/*	Select the system oscillator as sys_pllclkin	*/
	LPC_SYSCON->SYSPLLCLKSEL = 0x1;
	LPC_SYSCON->SYSPLLCLKUEN = 0;
	LPC_SYSCON->SYSPLLCLKUEN = 1;
	while ( !(LPC_SYSCON->SYSPLLCLKUEN & 0x01) ); /* Wait until updated */
#endif /* SYS_OSC */


#if ENABLE_CLKOUT
	/* Output the Clk onto the CLKOUT Pin PIO0_1 to monitor the freq on a scope */
	LPC_IOCON->PIO0_1	= (1<<0);
	/* Select the MAIN clock as the clock out selection since it's driving the core */
	LPC_SYSCON->CLKOUTCLKSEL = 3;
	/* Set CLKOUTDIV to 6 */
	LPC_SYSCON->CLKOUTDIV = 1;		//	CLKOUT Divider = 10
	/* Enable CLKOUT */
	LPC_SYSCON->CLKOUTUEN = 0;
	LPC_SYSCON->CLKOUTUEN = 1;
	while (!(LPC_SYSCON->CLKOUTUEN & 0x01));
#endif /* ENABLE_CLKOUT */


	profile.new_system_clock =	12000000UL;		//in MHZ
	profile.current_system_clock = 12000000UL;	//in MHz
	profile.power_mode = PARAM_LOW_CURRENT;

	if(config_pll_power(&profile)){
		SystemInit();
		UARTInit(UART_BAUD);
		printf("ROM PLL/PWR Config Fail\n\r");
		while(1);			
	}else{
		SystemAHBFrequency = profile.current_system_clock * LPC_SYSCON->SYSAHBCLKDIV;
		UARTInit(UART_BAUD);
		printf("ROM PLL/PWR Config Success\n\r");
	}

#if SYS_OSC
	/*	Turn off the IRC, BOD, ADC & WDT */
	LPC_SYSCON->PDRUNCFG |= (1<<0)|(1<<1)|(1<<3)|(1<<4)|(1<<6);
#else
	/*	Turn off the      BOD, ADC & WDT */
	LPC_SYSCON->PDRUNCFG |=               (1<<3)|(1<<4)|(1<<6);
#endif /* SYS_OSC */

	print_menu();

	LPC_GPIO2->DIR  |=   0xFF;
	LPC_GPIO2->DATA &= ~(0xFF);
	while(1){
	
		/*	Check if something was received on the UART */
		if(UARTCount){
			
			cmd = gchar();
			profile.new_system_clock = lookup_freq(cmd);

			/*	Apply the new frequency via the power profile API call */
			if(config_pll_power(&profile)){
				SystemAHBFrequency = profile.current_system_clock * LPC_SYSCON->SYSAHBCLKDIV;
				UARTInit(UART_BAUD);
				printf("ROM PLL/PWR Config Fail\nTry Again\n\r");			
			}else{
				SystemAHBFrequency = profile.current_system_clock * LPC_SYSCON->SYSAHBCLKDIV;
				UARTInit(UART_BAUD);
				printf("Selected freq: \x1b[4m%8d\x1b[0m | Newly configured freq: \x1b[5m%8d\x1b[0m Hz\r", profile.new_system_clock, profile.current_system_clock);
			}
		}

		/*	uC is alive indicator */
		if(counter == 0x10000){
			counter = 0;
			LPC_GPIO2->DATA ^= (1<<0);	//Toggle LED
		}else
			counter++;

		
	}
}


/*	Function used for print statements */
int  schar(int ch){
	uint8_t chr = ch;
	UARTSend(&chr, 1);
	return 0;
}

/* Function returns 1st character from the received UARTBuffer */
int32_t gchar(void) {
	uint8_t temp;

	while ( UARTCount == 0 );
	
	LPC_UART->IER = IER_THRE | IER_RLS;			/* Disable RBR */
	  //UARTSend( (uint8_t *)UARTBuffer, UARTCount );
	temp = UARTBuffer[0];
	UARTCount = 0;
	LPC_UART->IER = IER_THRE | IER_RLS | IER_RBR;	/* Re-enable RBR */
	return (int32_t)temp;
}
